#pragma once

#include <functional>
#include <QList>
#include <QDebug>
#include <QThread>
#include <QMutex>
#include <QMutexLocker>
#include <QWaitCondition>
#include <QScopedPointer>
#include <QSharedPointer>

#include "try.h"


/*

This class can be used to get/set a value in a thread-safe way.

If get is called before set, then it blocks until set is called in an other thread.

*/
template <class T> class Future
{
	private:
		QScopedPointer<T> data;
		QMutex mutex;
		QWaitCondition wait;
	
	public:
		void set (const T& value) {
			QMutexLocker lock(&mutex);
			data.reset(new T(value));
			wait.wakeAll();
		}
		
		void reset () {
			QMutexLocker lock(&mutex);
			data.reset();
		}

		T get () {
			QMutexLocker lock(&mutex);
			while (!data) wait.wait(&mutex);
			return *data;
		}
		
		QSharedPointer<T> peek () {
			QMutexLocker lock(&mutex);
			return QSharedPointer<T>(data ? new T(*data) : nullptr);
		}
		
		T getAndReset () {
			QMutexLocker lock(&mutex);
			while (!data) wait.wait(&mutex);
			T value = *data;
			data.reset();
			return value;
		}
		
		QSharedPointer<T> peekAndReset () {
			QMutexLocker lock(&mutex);
			return QSharedPointer<T>(data.take());
		}
};


template <class OBJ, class T> QSharedPointer<Future<Try<T>>> executeOnObject (OBJ *obj, std::function<T(OBJ*)> function) {
	// Check if thread is running to make sure the program
	// doesn't hang if someone forgot to start the thread.
	Q_ASSERT(obj->thread()->isRunning());
	
	// Create future
	auto future = QSharedPointer<Future<Try<T>>>::create();

	// Queue execution in other thread (execute immediately if current thread)
	QMetaObject::invokeMethod(obj, [=]{
		try {
			future->set(Try<T>::success(function(obj)));
		}
		catch (...) {
			future->set(Try<T>::failure(std::current_exception()));
		}
	});

	return future;
};

template <class OBJ> QSharedPointer<Future<Try<void>>> executeOnObject (OBJ *obj, std::function<void(OBJ*)> function) {
	// Check if thread is running to make sure the program
	// doesn't hang if someone forgot to start the thread.
	Q_ASSERT(obj->thread()->isRunning());
	
	// Create future
	auto future = QSharedPointer<Future<Try<void>>>::create();

	// Queue execution in other thread (execute immediately if current thread)
	QMetaObject::invokeMethod(obj, [=]{
		try {
			function(obj);
			future->set(Try<void>::success());
		}
		catch (...) {
			future->set(Try<void>::failure(std::current_exception()));
		}
	});

	return future;
};

template <class OBJ, class T> T executeOnObjectAndWait (OBJ *obj, std::function<T(OBJ*)> function) {
	return executeOnObject(obj, function)->get().getValue();
};

template <class OBJ> void executeOnObjectAndWait (OBJ *obj, std::function<void(OBJ*)> function) {
	executeOnObject(obj, function)->get().getValue();
};



/*

Start executing a function in a thread and return immediately. (blocks if thread is the current thread)
Exceptions are caught in the other thread and put in the Try object.

*/
template <class T> QSharedPointer<Future<Try<T>>> executeInThread (QThread *thread, std::function<T()> function) {
	// Create object in thread
	QObject *obj = new QObject;
	obj->moveToThread(thread);
	
	// Execute on object
	return executeOnObject<QObject, T>(obj, [=](QObject *obj){
		delete obj;
		return function();
	});
};

template <> inline QSharedPointer<Future<Try<void>>> executeInThread (QThread *thread, std::function<void()> function) {
	// Create object in thread
	QObject *obj = new QObject;
	obj->moveToThread(thread);
	
	// Execute on object
	return executeOnObject<QObject>(obj, [=](QObject *obj){
		delete obj;
		function();
	});
};



/*

Execute a function in a thread and wait for the result.

This also works with exceptions:
The exception is caught in the other thread and rethrown in this thread.

*/
template <class T> T executeInThreadAndWait (QThread *thread, std::function<T()> function) {
	return executeInThread<T>(thread, function)->get().getValue();
};

template <> inline void executeInThreadAndWait (QThread *thread, std::function<void()> function) {
	return executeInThread<void>(thread, function)->get().getValue();
};



template <class T> QList<T> getFutureValues (const QList<QSharedPointer<Future<T>>>& futures) {
	QList<T> values;
	for (const QSharedPointer<Future<T>>& future : futures) {
		values.append(future->get());
	}
	return values;
}


/*

Execute a function on a bunch of objects in their threads and wait for the result.

OBJ needs to be a subclass of QObject.

Exceptions are caught.

*/
template <class OBJ, class T> QList<Try<T>> executeInThreads (const QList<OBJ*>& objects, std::function<T(OBJ*)> function) {
	// Queue calls in other threads
	QList<QSharedPointer<Future<Try<T>>>> futures;

	for (OBJ* obj : objects) {
		futures << executeOnObject<OBJ, T>(obj, function);
	}

	// Collect data and return
	return getFutureValues(futures);
}

template <class OBJ> QList<Try<void>> executeInThreads (const QList<OBJ*>& objects, std::function<void(OBJ*)> function) {
	// Queue calls in other threads
	QList<QSharedPointer<Future<Try<void>>>> futures;

	for (OBJ* obj : objects) {
		futures << executeOnObject<OBJ>(obj, function);
	}

	// Collect data and return
	return getFutureValues(futures);
}
